<?php
/**
 * Hilfsfunktionen für ECONTO Website
 * Template-System für flexible Seiten
 */

/**
 * Rendert die Navigation aus dem Content-Array
 */
function renderNavigation($navArray) {
    // Sicherheitsprüfung: Ist es ein Array und nicht leer?
    if (empty($navArray) || !is_array($navArray)) {
        return '<ul></ul>';
    }
    
    $html = '<ul>';
    foreach ($navArray as $key => $navItem) {
        // Prüfen, ob navItem existiert und ein Array ist
        if (!isset($navItem['text']) || empty($navItem['text'])) {
            continue;
        }
        
        $first_letter = substr($navItem['text'], 0, 1);
        $rest_text = substr($navItem['text'], 1);
        
        // URL bestimmen: entweder spezifische URL oder Anchor-Link
        $url = isset($navItem['url']) ? htmlspecialchars($navItem['url']) : '#' . $key;
        
        $html .= '
        <li>
            <a href="' . $url . '">
                <span class="nav-first-letter">' . htmlspecialchars($first_letter) . '</span>' . htmlspecialchars($rest_text) . '
            </a>
        </li>';
    }
    $html .= '</ul>';
    return $html;
}

/**
 * Rendert eine Karte basierend auf Typ und Daten
 * Unterstützt: 'feature' und 'project'
 */
function renderCard($type, $data) {
    // Grundlegende Validierung
    if (empty($data) || !is_array($data) || !isset($data['title'])) {
        return '';
    }
    
    if ($type === 'feature') {
        return renderFeatureCard($data); // Nutzt die aktualisierte Funktion
    } elseif ($type === 'project') {
        return renderProjectCard($data);
    }
    
    return '';
}

/**
 * Rendert eine Feature-Karte (mit optionalem Link)
 */
function renderFeatureCard($data) {
    // Prüfen ob benötigte Daten vorhanden sind
    if (!isset($data['title']) || !isset($data['description'])) {
        return '';
    }
    
    $id = isset($data['id']) ? 'id="' . htmlspecialchars($data['id']) . '"' : '';
    $icon = isset($data['icon']) ? $data['icon'] : '';
    $has_link = !empty($data['link_url']);
    
    $card_content = '
        <div class="feature-card" ' . $id . '>
            ' . (!empty($icon) ? '
            <div class="feature-icon">
                <svg viewBox="0 0 24 24">
                    <path d="' . htmlspecialchars($icon) . '"/>
                </svg>
            </div>' : '') . '
            <h3>' . htmlspecialchars($data['title']) . '</h3>
            <p>' . htmlspecialchars($data['description']) . '</p>';
    
    // Optionaler Link für Feature-Karten
    if ($has_link) {
        $link_url = htmlspecialchars($data['link_url']);
        $link_text = isset($data['link_text']) ? htmlspecialchars($data['link_text']) : 'Mehr erfahren';
        $link_target = isset($data['link_target']) ? ' target="' . htmlspecialchars($data['link_target']) . '"' : '';
        
        $card_content .= '
            <a href="' . $link_url . '" class="feature-link"' . $link_target . '>
                ' . $link_text . ' <span class="arrow">→</span>
            </a>';
    }
    
    $card_content .= '
        </div>';
    
    return $card_content;
}

/**
 * Rendert eine Projekt-Karte mit optionaler Aufklappfunktion oder externem Link
 */
function renderProjectCard($data) {
    // Prüfen ob benötigte Daten vorhanden sind
    if (!isset($data['title']) || !isset($data['short_description'])) {
        return '';
    }
    
    $id = isset($data['id']) ? 'id="' . htmlspecialchars($data['id']) . '"' : '';
    $target_id = isset($data['id']) ? $data['id'] . '-text' : '';
    $icon = isset($data['icon']) ? $data['icon'] : '';
    $button_text = isset($data['button_text']) ? $data['button_text'] : 'Mehr erfahren';
    
    // Bestimmen, welcher Button-Typ verwendet wird
    $has_expanded_content = !empty($data['expanded_content']);
    $has_button_url = !empty($data['button_url']);
    $button_html = '';
    
    if ($has_button_url) {
        // Externer Link-Button
        $button_url = htmlspecialchars($data['button_url']);
        $button_target = isset($data['button_target']) ? ' target="' . htmlspecialchars($data['button_target']) . '"' : '';
        $button_html = '
            <a href="' . $button_url . '" class="project-link external-link"' . $button_target . '>
                ' . htmlspecialchars($button_text) . ' <span class="arrow">→</span>
            </a>';
    } elseif ($has_expanded_content) {
        // Aufklapp-Button
        $button_html = '
            <button class="project-link toggle-expand" data-target="' . $target_id . '">
                ' . htmlspecialchars($button_text) . ' <span class="arrow">→</span>
            </button>
            <div id="' . $target_id . '" class="expanded-text">
                ' . $data['expanded_content'] . '
            </div>';
    } elseif (isset($data['button_text'])) {
        // Fallback: Einfacher Button ohne Funktion (oder für JavaScript)
        $button_html = '
            <button class="project-link" onclick="return false;">
                ' . htmlspecialchars($button_text) . ' <span class="arrow">→</span>
            </button>';
    }
    
    return '
    <div class="project-card" ' . $id . '>
        ' . (!empty($icon) ? '
        <div class="project-image">
            <svg viewBox="0 0 24 24">
                <path d="' . htmlspecialchars($icon) . '"/>
            </svg>
        </div>' : '') . '
        <div class="project-content">
            <h3>' . htmlspecialchars($data['title']) . '</h3>
            <p>' . htmlspecialchars($data['short_description']) . '</p>
            ' . $button_html . '
        </div>
    </div>';
}

/**
 * Rendert Inhaltsabschnitte für Unterseiten
 */
function renderContentSections($sections) {
    if (empty($sections) || !is_array($sections)) {
        return '';
    }
    
    $html = '';
    foreach ($sections as $section) {
        if (!isset($section['title']) || !isset($section['content'])) {
            continue;
        }
        
        $html .= '
        <div class="content-section">
            <h3>' . htmlspecialchars($section['title']) . '</h3>
            <div class="section-content">
                ' . $section['content'] . '
            </div>
        </div>';
    }
    return $html;
}

// Erweiterte renderBreadcrumb() Funktion
function renderBreadcrumb($breadcrumb) {
    if (empty($breadcrumb)) {
        return '';
    }
    
    // Prüfen ob es ein Array oder String ist
    if (is_string($breadcrumb)) {
        return renderSimpleBreadcrumb($breadcrumb);
    }
    
    if (is_array($breadcrumb)) {
        return renderArrayBreadcrumb($breadcrumb);
    }
    
    return '';
}

function renderArrayBreadcrumb($items) {
    $html = '<div class="breadcrumb">';
    
    foreach ($items as $index => $item) {
        $is_last = ($index === count($items) - 1);
        
        if ($is_last || empty($item['url'])) {
            // Aktuelle Seite oder kein Link
            $html .= '<span class="' . ($is_last ? 'current' : '') . '">' 
                   . htmlspecialchars($item['text']) . '</span>';
        } else {
            // Link
            $html .= '<a href="' . htmlspecialchars($item['url']) . '">' 
                   . htmlspecialchars($item['text']) . '</a>';
        }
        
        // Trennzeichen (nicht nach dem letzten)
        if (!$is_last) {
            $html .= ' <span class="separator">›</span> ';
        }
    }
    
    $html .= '</div>';
    return $html;
}

/**
 * Debug-Funktion: Zeigt das Content-Array an (nur für Entwicklung)
 */
function debugContent($contentArray) {
    echo '<pre>';
    print_r($contentArray);
    echo '</pre>';
}

/**
 * Prüft, ob ein Content-Block existiert und Inhalt hat
 * Hilfsfunktion für komplexere Prüfungen
 */
function hasContent($array, $key) {
    return isset($array[$key]) && !empty($array[$key]);
}

/**
 * Rendert den Newsletter-Bereich
 */
function renderNewsletter($newsletterData) {
    // Prüfen, ob Newsletter-Daten vorhanden und gültig sind
    if (empty($newsletterData) || !is_array($newsletterData) || !isset($newsletterData['title'])) {
        return '';
    }
    
    $title = htmlspecialchars($newsletterData['title']);
    $description = isset($newsletterData['description']) ? htmlspecialchars($newsletterData['description']) : '';
    $placeholder = isset($newsletterData['placeholder']) ? htmlspecialchars($newsletterData['placeholder']) : 'Ihre E-Mail-Adresse';
    $button = isset($newsletterData['button']) ? htmlspecialchars($newsletterData['button']) : 'Abonnieren';
    $id = isset($newsletterData['id']) ? 'id="' . htmlspecialchars($newsletterData['id']) . '"' : 'id="contact"';
    
    $html = '
    <section class="newsletter" ' . $id . '>
        <div class="container">
            <h2>' . $title . '</h2>';
    
    if (!empty($description)) {
        $html .= '<p>' . $description . '</p>';
    }
    
    $html .= '
            <form class="newsletter-form" action="#" method="POST">
                <input type="email" name="email" class="newsletter-input" placeholder="' . $placeholder . '" required>
                <button type="submit" class="btn btn-primary">' . $button . '</button>
            </form>
        </div>
    </section>';
    
    return $html;
}

/**
 * Rendert das Kontaktformular
 */
function renderContactForm($data) {
    $html = '
    <section class="newsletter" id="contact">
        <div class="container">
            <h2>' . htmlspecialchars($data['title']) . '</h2>
            <p class="contact-description">' . $data['description'] . '</p>
            
            <form class="contact-form" id="contactForm" novalidate>';
    
    // Nur wenn fields existiert und ein Array ist, werden die Felder gerendert
    if (isset($data['fields']) && is_array($data['fields'])) {
        $html .= '<div class="form-grid">';
        
        // Felder rendern
        foreach ($data['fields'] as $fieldName => $field) {
            $required = isset($field['required']) && $field['required'] ? 'required' : '';
            $required_asterisk = $required ? '<span class="required">*</span>' : '';
            
            if ($field['type'] === 'textarea') {
                $html .= '
                        <div class="form-group full-width">
                            <label for="' . $fieldName . '">' . htmlspecialchars($field['placeholder']) . ' ' . $required_asterisk . '</label>
                            <textarea id="' . $fieldName . '" name="' . $fieldName . '" rows="' . $field['rows'] . '" placeholder="' . htmlspecialchars($field['placeholder']) . '" ' . $required . '></textarea>
                            <div class="error-message" id="' . $fieldName . '-error"></div>
                        </div>';
            } elseif ($field['type'] === 'select') {
                $html .= '
                        <div class="form-group">
                            <label for="' . $fieldName . '">' . htmlspecialchars($field['placeholder']) . ' ' . $required_asterisk . '</label>
                            <select id="' . $fieldName . '" name="' . $fieldName . '" ' . $required . '>';
                
                foreach ($field['options'] as $value => $label) {
                    $html .= '<option value="' . htmlspecialchars($value) . '">' . htmlspecialchars($label) . '</option>';
                }
                
                $html .= '
                            </select>
                            <div class="error-message" id="' . $fieldName . '-error"></div>
                        </div>';
            } else {
                $html .= '
                        <div class="form-group">
                            <label for="' . $fieldName . '">' . htmlspecialchars($field['placeholder']) . ' ' . $required_asterisk . '</label>
                            <input type="' . $field['type'] . '" id="' . $fieldName . '" name="' . $fieldName . '" placeholder="' . htmlspecialchars($field['placeholder']) . '" ' . $required . '>
                            <div class="error-message" id="' . $fieldName . '-error"></div>
                        </div>';
            }
        }
        
        $html .= '</div>';
    }
    
    // Privacy-Checkbox nur anzeigen, wenn Text vorhanden ist
    if (!empty($data['privacy_text'])) {
        $html .= '
                <div class="form-group full-width">
                    <div class="privacy-checkbox">
                        <input type="checkbox" id="privacy" name="privacy" required>
                        <label for="privacy">' . $data['privacy_text'] . '</label>
                        <div class="error-message" id="privacy-error"></div>
                    </div>
                </div>';
    }
    
    // Button nur anzeigen, wenn Text vorhanden ist
    if (!empty($data['button'])) {
        $html .= '
                <div class="form-submit">
                    <button type="submit" class="btn btn-primary contact-submit">
                        ' . htmlspecialchars($data['button']) . '
                        <span class="submit-spinner" style="display: none;">⏳</span>
                    </button>
                    <div class="form-message" id="formMessage"></div>
                </div>';
    }
    
    $html .= '
            </form>
        </div>
    </section>';
    
    return $html;
}

function renderFooter($ContentArray) {
    // Prüfe, ob der Footer-Bereich existiert
    if (!isset($ContentArray['footer']) || empty($ContentArray['footer'])) {
        return '';
    }

    $footer = $ContentArray['footer'];
    
    $html = '
    <footer>
        <div class="container">
            <div class="footer-content">';

    // Alle Spalten durchlaufen (about, explore, contact)
    $columns = ['about', 'explore', 'contact'];
    
    foreach ($columns as $columnKey) {
        if (!isset($footer[$columnKey]) || empty($footer[$columnKey])) {
            continue;
        }
        
        $column = $footer[$columnKey];
        $html .= '
                <div class="footer-column">
                    <h3>' . htmlspecialchars($column['title']) . '</h3>';
        
        // Description rendern, falls vorhanden
        if (isset($column['description']) && !empty($column['description'])) {
            $html .= '
                    <p>' . htmlspecialchars($column['description']) . '</p>';
        }
        
        // Links/Details rendern, falls vorhanden
        if (isset($column['details']) && is_array($column['details']) && !empty($column['details'])) {
            $html .= '
                    <ul class="footer-links">';
            
            foreach ($column['details'] as $detail) {
                $html .= renderFooterEntry($detail);
            }
            
            $html .= '
                    </ul>';
        }
        
        $html .= '
                </div>';
    }

    $html .= '
            </div>
            
            <div class="copyright">
                <p>' . (isset($footer['copyright']) ? $footer['copyright'] : '&copy; ' . date('Y') . ' ECONTO<sup>®</sup> Controlling') . '</p>';
    
    if (isset($footer['disclaimer']) && !empty($footer['disclaimer'])) {
        $html .= '
                <p>' . htmlspecialchars($footer['disclaimer']) . '</p>';
    }
    
    $html .= '
            </div>
        </div>
    </footer>';

    return $html;
}

/**
 * Rendert einen Footer-Eintrag (Link oder Text)
 * Unterstützt verschiedene Strukturen:
 * - Mit URL (für Links): array('url' => 'mailto:...', 'text' => '...')
 * - Ohne URL (für reine Text): array('text' => '...')
 * - Mit Label und Wert: array('label' => 'Sprache:', 'value' => 'php')
 */
function renderFooterEntry($entry) {
    // Prüfen, welche Datenstruktur verwendet wird
    if (isset($entry['label']) && isset($entry['value'])) {
        // Struktur mit Label/Value (für explore)
        return '
        <li>
            <span class="footer-label">' . htmlspecialchars($entry['label']) . '</span> ' . 
            htmlspecialchars($entry['value']) . '
        </li>';
    } 
    elseif (isset($entry['url']) && !empty($entry['url'])) {
        // URL vorhanden -> Link
        return '
        <li>
            <a href="' . htmlspecialchars($entry['url']) . '">' . 
            htmlspecialchars($entry['text']) . '</a>
        </li>';
    }
    elseif (isset($entry['text'])) {
        // Nur Text
        return '
        <li>' . htmlspecialchars($entry['text']) . '</li>';
    }
    
    return '';
}

/**
 * Rendert einen Footer-Link oder Text
 */
function renderFooterLink($link) {
    if (!isset($link['text']) || empty($link['text'])) {
        return '';
    }
    
    $first_letter = substr($link['text'], 0, 1);
    $rest_text = substr($link['text'], 1);
    
    if (!empty($link['url'])) {
        return '
        <li>
            <a href="' . htmlspecialchars($link['url']) . '">
                <span class="nav-first-letter">' . htmlspecialchars($first_letter) . '</span>' . htmlspecialchars($rest_text) . '
            </a>
        </li>';
    } else {
        return '
        <li>
            <span>' . htmlspecialchars($link['text']) . '</span>
        </li>';
    }
}
?>
