/**
 * ECONTO Website JavaScript
 * Template-System unterstützende Funktionen
 */

// Mobile Navigation Toggle
const mobileMenuBtn = document.getElementById('mobileMenuBtn');
const mainNav = document.getElementById('mainNav');

if (mobileMenuBtn && mainNav) {
    mobileMenuBtn.addEventListener('click', () => {
        mainNav.classList.toggle('active');
        mobileMenuBtn.classList.toggle('active');
    });
    
    // Schließe Mobile Menu bei Klick auf einen Link
    const navLinks = document.querySelectorAll('nav a');
    navLinks.forEach(link => {
        link.addEventListener('click', () => {
            mainNav.classList.remove('active');
            mobileMenuBtn.classList.remove('active');
        });
    });
}

// Header Hintergrund bei Scrollen ändern
window.addEventListener('scroll', () => {
    const header = document.querySelector('header');
    if (header) {
        if (window.scrollY > 50) {
            header.style.padding = '15px 0';
            header.style.boxShadow = '0 5px 20px rgba(0, 0, 0, 0.1)';
        } else {
            header.style.padding = '20px 0';
            header.style.boxShadow = '0 2px 10px rgba(0, 0, 0, 0.08)';
        }
    }
});

// Intelligente Navigation für alle Links
function initSmartNavigation() {
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function (e) {
            const href = this.getAttribute('href');
            
            // Nur für interne Anchor-Links
            if (href !== '#' && href.startsWith('#')) {
                e.preventDefault();
                
                const targetElement = document.querySelector(href);
                if (targetElement) {
                    // Prüfe den Typ des Ziels
                    const isProjectCard = targetElement.classList.contains('project-card');
                    const isFeatureCard = targetElement.classList.contains('feature-card');
                    
                    // Bei Projekt-Karten: Öffne wenn möglich
                    if (isProjectCard) {
                        const expandButton = targetElement.querySelector('.toggle-expand');
                        const expandedText = document.getElementById(href.replace('#', '') + '-text');
                        
                        // Öffne den erweiterten Text, wenn er geschlossen ist
                        if (expandButton && expandedText && !expandedText.classList.contains('active')) {
                            setTimeout(() => {
                                expandButton.click();
                            }, 500);
                        }
                    }
                    
                    // Bei Feature-Karten: Visuelles Feedback
                    if (isFeatureCard) {
                        targetElement.style.transform = 'translateY(-15px)';
                        setTimeout(() => {
                            targetElement.style.transform = 'translateY(-10px)';
                        }, 300);
                    }
                    
                    // Scroll zur Position
                    window.scrollTo({
                        top: targetElement.offsetTop - 100,
                        behavior: 'smooth'
                    });
                }
            }
        });
    });
}

// Aufklappfunktion für "Mehr erfahren" Buttons
function initExpandableCards() {
    document.querySelectorAll('.toggle-expand').forEach(button => {
        button.addEventListener('click', function() {
            const targetId = this.getAttribute('data-target');
            const expandedText = document.getElementById(targetId);
            
            if (expandedText) {
                // Schließe alle anderen geöffneten Projekt-Karten
                if (expandedText.classList.contains('active')) {
                    // Wenn bereits aktiv, schließe es
                    expandedText.classList.remove('active');
                    updateButtonText(this, false);
                } else {
                    // Sonst schließe alle anderen und öffne dieses
                    document.querySelectorAll('.expanded-text.active').forEach(expanded => {
                        expanded.classList.remove('active');
                        const otherButton = expanded.previousElementSibling;
                        if (otherButton && otherButton.classList.contains('toggle-expand')) {
                            updateButtonText(otherButton, false);
                        }
                    });
                    
                    expandedText.classList.add('active');
                    updateButtonText(this, true);
                }
            }
        });
    });
    
    function updateButtonText(button, isActive) {
        const arrow = button.querySelector('.arrow');
        if (isActive) {
            arrow.innerHTML = '↑';
            arrow.style.transform = 'rotate(0deg)';
            button.innerHTML = button.innerHTML.replace('Mehr erfahren', 'Weniger anzeigen');
        } else {
            arrow.innerHTML = '→';
            arrow.style.transform = 'rotate(0deg)';
            button.innerHTML = button.innerHTML.replace('Weniger anzeigen', 'Mehr erfahren');
        }
    }
}

// Hover-Effekt für Navigation Links mit hervorgehobenen Buchstaben
function initHoverEffects() {
    document.querySelectorAll('nav a').forEach(link => {
        link.addEventListener('mouseenter', function() {
            const firstLetter = this.querySelector('.nav-first-letter');
            if (firstLetter) {
                firstLetter.style.transform = 'scale(1.2) rotate(5deg)';
            }
        });
        
        link.addEventListener('mouseleave', function() {
            const firstLetter = this.querySelector('.nav-first-letter');
            if (firstLetter) {
                firstLetter.style.transform = 'scale(1) rotate(0deg)';
            }
        });
    });
}

// Highlight-Effekt für direkt anvisierte Karten
function initHashNavigation() {
    window.addEventListener('hashchange', function() {
        const hash = window.location.hash;
        if (hash) {
            const targetElement = document.querySelector(hash);
            if (targetElement) {
                // Prüfe ob es eine Projekt-Karte ist
                if (targetElement.classList.contains('project-card')) {
                    const expandButton = targetElement.querySelector('.toggle-expand');
                    const expandedText = document.getElementById(hash.replace('#', '') + '-text');
                    
                    if (expandButton && expandedText && !expandedText.classList.contains('active')) {
                        setTimeout(() => {
                            expandButton.click();
                        }, 500);
                    }
                }
                
                // Scroll zur Position
                setTimeout(() => {
                    targetElement.scrollIntoView({ behavior: 'smooth', block: 'start' });
                }, 100);
            }
        }
    });
}

// Newsletter Formular Handling
function initNewsletterForm() {
    const newsletterForm = document.querySelector('.newsletter-form');
    if (newsletterForm) {
        newsletterForm.addEventListener('submit', function(e) {
            e.preventDefault();
            const emailInput = this.querySelector('.newsletter-input');
            const email = emailInput.value.trim();
            
            if (email && validateEmail(email)) {
                // Hier AJAX-Anfrage oder Formular-Submit
                console.log('Newsletter-Anmeldung:', email);
                
                // Erfolgsmeldung anzeigen
                alert('Vielen Dank für Ihre Anmeldung! Wir haben eine Bestätigung an ' + email + ' gesendet.');
                emailInput.value = '';
            } else {
                alert('Bitte geben Sie eine gültige E-Mail-Adresse ein.');
                emailInput.focus();
            }
        });
    }
    
    function validateEmail(email) {
        const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    }
}

// Kontaktformular Handling (NEU)
function initContactForm() {
    const contactForm = document.getElementById('contactForm');
    if (!contactForm) return;
    
    // Fehlermeldungen zurücksetzen
    function resetErrors() {
        document.querySelectorAll('.error-message').forEach(el => {
            el.classList.remove('show');
            el.textContent = '';
        });
        document.querySelectorAll('.form-group input, .form-group select, .form-group textarea').forEach(el => {
            el.classList.remove('error');
        });
    }
    
    // Erfolgs-/Fehlermeldung anzeigen
    function showMessage(message, type = 'success') {
        const messageEl = document.getElementById('formMessage');
        if (!messageEl) return;
        
        messageEl.textContent = message;
        messageEl.className = 'form-message ' + type;
        messageEl.classList.add('show');
        
        // Nach 10 Sekunden ausblenden
        setTimeout(() => {
            messageEl.classList.remove('show');
        }, 10000);
    }
    
    // Formular validieren
    function validateForm(formData) {
        let isValid = true;
        resetErrors();
        
        // Name validieren
        if (!formData.get('name') || formData.get('name').trim() === '') {
            showError('name', 'Bitte geben Sie Ihren Namen ein.');
            isValid = false;
        }
        
        // Email validieren
        const email = formData.get('email');
        if (!email || email.trim() === '') {
            showError('email', 'Bitte geben Sie Ihre E-Mail-Adresse ein.');
            isValid = false;
        } else if (!validateEmail(email)) {
            showError('email', 'Bitte geben Sie eine gültige E-Mail-Adresse ein.');
            isValid = false;
        }
        
        // Betreff validieren
        if (!formData.get('subject') || formData.get('subject') === '') {
            showError('subject', 'Bitte wählen Sie einen Betreff aus.');
            isValid = false;
        }
        
        // Nachricht validieren
        if (!formData.get('message') || formData.get('message').trim() === '') {
            showError('message', 'Bitte geben Sie eine Nachricht ein.');
            isValid = false;
        } else if (formData.get('message').trim().length < 10) {
            showError('message', 'Ihre Nachricht sollte mindestens 10 Zeichen lang sein.');
            isValid = false;
        }
        
        // Datenschutz validieren
        if (!formData.get('privacy')) {
            showError('privacy', 'Bitte stimmen Sie der Datenschutzerklärung zu.');
            isValid = false;
        }
        
        return isValid;
    }
    
    // Fehler anzeigen
    function showError(fieldId, message) {
        const errorEl = document.getElementById(fieldId + '-error');
        const inputEl = document.getElementById(fieldId);
        
        if (errorEl) {
            errorEl.textContent = message;
            errorEl.classList.add('show');
        }
        
        if (inputEl) {
            inputEl.classList.add('error');
        }
    }
    
    // Email validieren
    function validateEmail(email) {
        const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    }
    
    // Formular absenden
    contactForm.addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const submitButton = this.querySelector('.contact-submit');
        const spinner = this.querySelector('.submit-spinner');
        const formData = new FormData(this);
        
        // Validierung
        if (!validateForm(formData)) {
            return;
        }
        
        // Button deaktivieren und Spinner anzeigen
        submitButton.disabled = true;
        spinner.style.display = 'inline-block';
        
        try {
            // Hier würdest du die Daten an deinen Server senden
            // Beispiel mit fetch:
            // const response = await fetch('contact-submit.php', {
            //     method: 'POST',
            //     body: formData
            // });
            
            // Simulierte Server-Antwort (für Demo)
            await new Promise(resolve => setTimeout(resolve, 1500));
            
            // Erfolgsmeldung
            showMessage('Vielen Dank für Ihre Nachricht! Wir werden uns in Kürze bei Ihnen melden.', 'success');
            
            // Formular zurücksetzen
            contactForm.reset();
            resetErrors();
            
        } catch (error) {
            console.error('Fehler beim Senden:', error);
            showMessage('Es ist ein Fehler aufgetreten. Bitte versuchen Sie es später erneut oder kontaktieren Sie uns direkt unter office@econto.at.', 'error');
        } finally {
            // Button wieder aktivieren und Spinner ausblenden
            submitButton.disabled = false;
            spinner.style.display = 'none';
        }
    });
    
    // Live-Validierung für Eingabefelder
    const inputs = contactForm.querySelectorAll('input, select, textarea');
    inputs.forEach(input => {
        input.addEventListener('blur', function() {
            if (this.value.trim() !== '') {
                const errorEl = document.getElementById(this.id + '-error');
                if (errorEl) {
                    errorEl.classList.remove('show');
                }
                this.classList.remove('error');
            }
        });
        
        // Für Textarea zusätzlich Zeichenzählung
        if (input.type === 'textarea') {
            input.addEventListener('input', function() {
                const errorEl = document.getElementById(this.id + '-error');
                if (this.value.trim().length >= 10 && errorEl) {
                    errorEl.classList.remove('show');
                    this.classList.remove('error');
                }
            });
        }
        
        // Für Email-Feld zusätzliche Validierung
        if (input.type === 'email') {
            input.addEventListener('input', function() {
                if (validateEmail(this.value)) {
                    const errorEl = document.getElementById(this.id + '-error');
                    if (errorEl) {
                        errorEl.classList.remove('show');
                    }
                    this.classList.remove('error');
                }
            });
        }
    });
}

// Initialisierung beim Laden der Seite
document.addEventListener('DOMContentLoaded', function() {
    // Alle Funktionen initialisieren
    initSmartNavigation();
    initExpandableCards();
    initHoverEffects();
    initHashNavigation();
    initNewsletterForm();
    initContactForm();
    
    // Prüfe ob eine direkte URL mit Hash geladen wurde
    const hash = window.location.hash;
    if (hash) {
        setTimeout(() => {
            const targetElement = document.querySelector(hash);
            if (targetElement) {
                // Wenn es eine Projekt-Karte ist, öffne sie
                if (targetElement.classList.contains('project-card')) {
                    const expandButton = targetElement.querySelector('.toggle-expand');
                    const expandedText = document.getElementById(hash.replace('#', '') + '-text');
                    
                    if (expandButton && expandedText && !expandedText.classList.contains('active')) {
                        setTimeout(() => {
                            expandButton.click();
                        }, 1000);
                    }
                }
                
                // Scroll zur Position
                setTimeout(() => {
                    targetElement.scrollIntoView({ behavior: 'smooth', block: 'start' });
                }, 500);
            }
        }, 500);
    }
});
