<?php

/**
 * Kopiert Dateien aus dem aktuellen Verzeichnis in ausgewählte oder alle Unterverzeichnisse
 * basierend auf Include- und Exclude-Mustern.
 * 
 * @param string|array|null $includePatterns Dateimuster für Dateien, die KOPIERT werden sollen
 *                                           (Standard: null = keine Dateien)
 * @param string|array $excludePatterns      Dateimuster für Dateien, die NICHT kopiert werden sollen
 *                                           (Standard: '*.*' = alle Dateien ausschließen)
 * @param string|array $targetSubdirs        Spezifische Unterverzeichnisse, in die kopiert werden soll
 *                                           (Standard: "" = alle Unterverzeichnisse)
 * 
 * @return array Statistik-Array mit Schlüsseln:
 *              - 'copied': Anzahl erfolgreich kopierter/aktualisierter Dateien
 *              - 'skipped': Anzahl übersprungener Dateien (Filter, nicht neuer, etc.)
 *              - 'errors': Anzahl aufgetretener Fehler
 *              - 'total_processed': Gesamtzahl verarbeiteter Datei-Operationen
 *              - 'details': Array mit detaillierten Meldungen
 *              - 'target_dirs_processed': Anzahl verarbeiteter Zielverzeichnisse
 * 
 * @throws RuntimeException Wenn aktuelles Verzeichnis nicht lesbar
 * @throws InvalidArgumentException Wenn Parameter nicht string oder array sind
 * 
 * ## Beispiele:
 * 
 * // 1. Standard: Ohne Parameter wird nichts kopiert
 * copyFilesToSubdirectories();
 * 
 * // 2. In alle Unterverzeichnisse kopieren (wie bisher)
 * copyFilesToSubdirectories('*.*', [], '');
 * 
 * // 3. Nur in spezifische Unterverzeichnisse kopieren
 * copyFilesToSubdirectories('*.php', [], 'subdir1');
 * 
 * // 4. In mehrere spezifische Unterverzeichnisse kopieren
 * copyFilesToSubdirectories('*.php', [], ['subdir1', 'subdir2', 'images']);
 * 
 * // 5. Komplexes Beispiel mit allen Parametern
 * copyFilesToSubdirectories(
 *     ['*.php', '*.css', '*.js'],  // Include
 *     ['content.php', 'test*.php'], // Exclude
 *     ['public', 'admin', 'uploads'] // Target directories
 * );
 */
function copyFilesToSubdirectories(
    $includePatterns = null,
    $excludePatterns = '*.*',
    $targetSubdirs = ""
): array {
    $currentDir = __DIR__;
    $results = [
        'copied' => 0,
        'skipped' => 0,
        'errors' => 0,
        'total_processed' => 0,
        'target_dirs_processed' => 0,
        'details' => [],
        'warning' => ''
    ];
    
    try {
        // Warnung, wenn includePatterns null ist
        if ($includePatterns === null) {
            $results['warning'] = 'Keine Include-Muster angegeben. Standardmäßig wird nichts kopiert (Sicherheitsfeature).';
            $results['details'][] = $results['warning'];
            return $results;
        }
        
        // Parameter normalisieren
        $includePatterns = normalizePatterns($includePatterns);
        $excludePatterns = normalizePatterns($excludePatterns);
        
        // Zielverzeichnisse normalisieren
        $targetDirs = normalizeTargetSubdirs($targetSubdirs, $currentDir);
        
        // Verzeichnis prüfen
        if (!is_readable($currentDir) || !is_dir($currentDir)) {
            throw new RuntimeException("Verzeichnis '$currentDir' ist nicht lesbar oder existiert nicht.");
        }
        
        // Alle Einträge im aktuellen Verzeichnis lesen
        $entries = scandir($currentDir);
        
        // Dateien im aktuellen Verzeichnis sammeln
        $files = [];
        foreach ($entries as $entry) {
            $entryPath = $currentDir . DIRECTORY_SEPARATOR . $entry;
            
            if ($entry !== '.' && $entry !== '..' && is_file($entryPath)) {
                $files[] = $entry;
            }
        }
        
        // Wenn targetDirs leer, alle Unterverzeichnisse finden
        if (empty($targetDirs)) {
            $targetDirs = [];
            foreach ($entries as $entry) {
                $entryPath = $currentDir . DIRECTORY_SEPARATOR . $entry;
                
                if ($entry !== '.' && $entry !== '..' && is_dir($entryPath)) {
                    $targetDirs[] = $entry;
                }
            }
        }
        
        // Für jedes Ziel-Unterverzeichnis die Dateien kopieren
        foreach ($targetDirs as $subdir) {
            $subdirPath = $currentDir . DIRECTORY_SEPARATOR . $subdir;
            
            // Prüfen, ob das Unterverzeichnis existiert
            if (!is_dir($subdirPath)) {
                $results['details'][] = "Warnung: Zielverzeichnis '$subdir' existiert nicht";
                continue;
            }
            
            // Prüfen, ob das Unterverzeichnis beschreibbar ist
            if (!is_writable($subdirPath)) {
                $results['details'][] = "Warnung: Verzeichnis '$subdir' ist nicht beschreibbar";
                continue;
            }
            
            $results['target_dirs_processed']++;
            
            foreach ($files as $file) {
                $results['total_processed']++;
                
                // Prüfen, ob die Datei den Mustern entspricht
                if (!shouldProcessFile($file, $includePatterns, $excludePatterns)) {
                    $results['skipped']++;
                    continue;
                }
                
                $sourcePath = $currentDir . DIRECTORY_SEPARATOR . $file;
                $destPath = $subdirPath . DIRECTORY_SEPARATOR . $file;
                
                // Kopiervorgang
                try {
                    if (file_exists($destPath)) {
                        // Nur kopieren, wenn die Quelldatei neuer ist
                        if (filemtime($sourcePath) > filemtime($destPath)) {
                            if (@copy($sourcePath, $destPath)) {
                                touch($destPath, filemtime($sourcePath));
                                $results['copied']++;
                                $results['details'][] = "Aktualisiert: $file -> $subdir/";
                            } else {
                                $results['errors']++;
                                $results['details'][] = "Fehler: $file -> $subdir/ (Kopieren fehlgeschlagen)";
                            }
                        } else {
                            $results['skipped']++;
                        }
                    } else {
                        // Datei existiert nicht im Ziel
                        if (@copy($sourcePath, $destPath)) {
                            $results['copied']++;
                            $results['details'][] = "Kopiert: $file -> $subdir/";
                        } else {
                            $results['errors']++;
                            $results['details'][] = "Fehler: $file -> $subdir/ (Kopieren fehlgeschlagen)";
                        }
                    }
                } catch (Exception $e) {
                    $results['errors']++;
                    $results['details'][] = "Fehler: $file -> $subdir/ ({$e->getMessage()})";
                }
            }
        }
        
        // Zusammenfassung
        if ($results['target_dirs_processed'] === 0) {
            $results['warning'] = 'Keine gültigen Zielverzeichnisse gefunden oder verarbeitet.';
        }
        
    } catch (Exception $e) {
        $results['errors']++;
        $results['details'][] = "Allgemeiner Fehler: " . $e->getMessage();
    }
    
    return $results;
}

/**
 * Normalisiert die Ziel-Unterverzeichnis-Parameter
 */
function normalizeTargetSubdirs($targetSubdirs, string $currentDir): array {
    // Wenn leerer String oder null, leeres Array zurückgeben
    if ($targetSubdirs === "" || $targetSubdirs === null) {
        return [];
    }
    
    // Wenn String, in Array umwandeln
    if (is_string($targetSubdirs)) {
        // Komma-getrennte Liste unterstützen
        if (strpos($targetSubdirs, ',') !== false) {
            $targetSubdirs = array_map('trim', explode(',', $targetSubdirs));
        } else {
            $targetSubdirs = [trim($targetSubdirs)];
        }
    }
    
    // Sicherstellen, dass es ein Array ist
    if (!is_array($targetSubdirs)) {
        throw new InvalidArgumentException('TargetSubdirs muss ein String oder Array sein');
    }
    
    // Leere Einträge entfernen und Duplikate eliminieren
    $targetSubdirs = array_filter($targetSubdirs, function($dir) {
        return $dir !== '' && $dir !== null;
    });
    
    $targetSubdirs = array_unique($targetSubdirs);
    
    // Sicherheitscheck: Kein Verzeichnis mit ".." erlauben
    foreach ($targetSubdirs as $dir) {
        if (strpos($dir, '..') !== false) {
            throw new InvalidArgumentException("Verzeichnisname '$dir' enthält ungültige Zeichen '..'");
        }
    }
    
    return array_values($targetSubdirs);
}

/**
 * Normalisiert Muster-Parameter (String oder Array zu Array)
 */
function normalizePatterns($patterns): array {
    if ($patterns === null) {
        return [];
    }
    
    if (is_string($patterns)) {
        // Komma-getrennte Liste unterstützen
        if (strpos($patterns, ',') !== false) {
            return array_map('trim', explode(',', $patterns));
        }
        return [$patterns];
    }
    
    if (is_array($patterns)) {
        return $patterns;
    }
    
    throw new InvalidArgumentException(
        'Muster müssen ein String oder Array sein. ' . gettype($patterns) . ' gegeben.'
    );
}

/**
 * Entscheidet, ob eine Datei verarbeitet werden soll basierend auf Include/Exclude-Mustern
 */
function shouldProcessFile(string $filename, array $includePatterns, array $excludePatterns): bool {
    // Wenn Include-Muster vorhanden sind, muss die Datei mindestens einem entsprechen
    if (!empty($includePatterns)) {
        $includeMatch = false;
        foreach ($includePatterns as $pattern) {
            if (fnmatch($pattern, $filename, FNM_CASEFOLD)) {
                $includeMatch = true;
                break;
            }
        }
        
        // Wenn kein Include-Muster passt, Datei ausschließen
        if (!$includeMatch) {
            return false;
        }
    }
    
    // Wenn Exclude-Muster vorhanden sind, darf die Datei keinem entsprechen
    if (!empty($excludePatterns)) {
        foreach ($excludePatterns as $pattern) {
            if (fnmatch($pattern, $filename, FNM_CASEFOLD)) {
                return false;
            }
        }
    }
    
    return true;
}

// ==================== BEISPIELVERWENDUNG ====================

/**
 * Beispielaufrufe für die erweiterte Funktion
 */
function testCopyFunctionExtended() {
    echo "=== Test 1: Standard (nichts kopieren) ===\n";
    $result = copyFilesToSubdirectories();
    print_r($result);
    
    echo "\n=== Test 2: Alle Dateien in alle Unterverzeichnisse ===\n";
    $result = copyFilesToSubdirectories('*.*', [], '');
    print_r($result);
    
    echo "\n=== Test 3: PHP-Dateien in spezifisches Verzeichnis ===\n";
    $result = copyFilesToSubdirectories('*.php', [], 'subdir1');
    print_r($result);
    
    echo "\n=== Test 4: In mehrere Verzeichnisse ===\n";
    $result = copyFilesToSubdirectories('*.php', [], ['public', 'admin', 'includes']);
    print_r($result);
    
    echo "\n=== Test 5: Komma-getrennte Liste von Verzeichnissen ===\n";
    $result = copyFilesToSubdirectories('*.php', [], 'dir1, dir2, dir3');
    print_r($result);
    
    echo "\n=== Test 6: Komplexes Beispiel ===\n";
    $result = copyFilesToSubdirectories(
        ['*.php', '*.css', '*.js'],  // Include
        ['content.php', 'test*.php'], // Exclude
        ['public', 'admin']           // Target directories
    );
    print_r($result);
    
    echo "\n=== Test 7: Nur bestimmte Dateien in bestimmte Verzeichnisse ===\n";
    $result = copyFilesToSubdirectories(
        ['index.php', 'style.css', 'config.php'],
        [],
        ['backup', 'archive']
    );
    print_r($result);
}
?>
